<?php
// /user/stripe_webhook.php

// ✅ Turn OFF in production
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/stripe-webhook-error.log');

require __DIR__ . '/../vendor/autoload.php';

// Your Stripe secret key (same as other PHP files)
// \Stripe\Stripe::setApiKey('sk_test_51MbycxKqjaL0X49YF3I3ZZ81fF772xlz4XB90p4nDxdYFWR4CyxOljSTiZy26eS3V9dZFrNotiSA8maUnAlmQG5M00dZqUUuCS');
\Stripe\Stripe::setApiKey('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');

// 🔴 Webhook Signing Secret from Stripe Dashboard
$endpoint_secret = 'whsec_y28Gst2BCsD5YI6Ck3KjMF9HyX1cOGZ3';

// Read raw body & Stripe signature header
$payload    = @file_get_contents('php://input');
$sig_header = $_SERVER['HTTP_STRIPE_SIGNATURE'] ?? '';
$event      = null;

if (!$payload) {
    http_response_code(400);
    echo 'No payload';
    exit;
}

try {
    // Verify webhook signature
    $event = \Stripe\Webhook::constructEvent(
        $payload,
        $sig_header,
        $endpoint_secret
    );
} catch (\UnexpectedValueException $e) {
    http_response_code(400);
    echo 'Invalid payload';
    exit;
} catch (\Stripe\Exception\SignatureVerificationException $e) {
    http_response_code(400);
    echo 'Invalid signature';
    exit;
}

// Connect to DB
require __DIR__ . '/../connect.php';
if (!isset($con) || !($con instanceof mysqli) || $con->connect_error) {
    error_log('DB connection error in webhook: ' . ($con->connect_error ?? 'no $con'));
    http_response_code(500);
    echo 'DB error';
    exit;
}

/**
 * Build a clean payment details array from PaymentIntent (+ Charge + Session).
 *
 * This returns a SINGLE event entry (we will aggregate events later).
 *
 * @param \Stripe\PaymentIntent|null      $pi
 * @param \Stripe\Checkout\Session|null   $session
 * @param string|null                     $eventType
 * @return array
 */
function buildPaymentDetails(?\Stripe\PaymentIntent $pi, ?\Stripe\Checkout\Session $session = null, ?string $eventType = null): array
{
    $details = [
        'source'        => 'stripe_webhook',
        'event_type'    => $eventType,
        'created_at'    => time(),
        'session'       => null,
        'payment_intent'=> null,
        'charge'        => null,
    ];

    if ($session) {
        $details['session'] = [
            'id'              => $session->id,
            'status'          => $session->status,
            'payment_status'  => $session->payment_status ?? null,
            'mode'            => $session->mode ?? null,
            'currency'        => $session->currency ?? null,
            'amount_total'    => isset($session->amount_total) ? $session->amount_total / 100 : null,
            'customer_email'  => $session->customer_details->email ?? null,
        ];
    }

    if ($pi) {
        $charge = null;
        if (isset($pi->charges)
            && isset($pi->charges->data)
            && count($pi->charges->data) > 0
        ) {
            $charge = $pi->charges->data[0];
        }

        $details['payment_intent'] = [
            'id'             => $pi->id,
            'amount'         => $pi->amount / 100,
            'currency'       => $pi->currency,
            'status'         => $pi->status,
            'created'        => $pi->created,
            'payment_method' => $pi->payment_method,
        ];

        if ($charge) {
            $card = $charge->payment_method_details->card ?? null;

            $details['charge'] = [
                'id'           => $charge->id,
                'status'       => $charge->status,
                'paid'         => $charge->paid,
                'amount'       => $charge->amount / 100,
                'currency'     => $charge->currency,
                'receipt_url'  => $charge->receipt_url,
                'card'         => $card ? [
                    'brand'     => $card->brand,
                    'last4'     => $card->last4,
                    'exp_month' => $card->exp_month,
                    'exp_year'  => $card->exp_year,
                    'funding'   => $card->funding,
                ] : null,
            ];
        }
    }

    return $details;
}

/**
 * Helper: merge new event details into existing payment_details JSON.
 *
 * Structure in DB will be:
 * {
 *   "events": [ {event1}, {event2}, ... ]
 * }
 *
 * @param string|null $existingJson
 * @param array|null  $newDetails  single event details (from buildPaymentDetails)
 * @return string|null             merged JSON or null
 */
function mergePaymentDetails(?string $existingJson, ?array $newDetails): ?string
{
    if ($newDetails === null) {
        return $existingJson; // nothing new
    }

    // If there is no existing details, just wrap this event
    if (empty($existingJson)) {
        $wrapper = [
            'events' => [$newDetails],
        ];
        return json_encode($wrapper, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    }

    $decoded = json_decode($existingJson, true);
    if (json_last_error() !== JSON_ERROR_NONE || !is_array($decoded)) {
        // Old / invalid JSON: turn old data into first event + new as second
        $wrapper = [
            'events' => [$decoded, $newDetails],
        ];
        return json_encode($wrapper, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    }

    // If it already has an 'events' array, append
    if (isset($decoded['events']) && is_array($decoded['events'])) {
        $decoded['events'][] = $newDetails;
    } else {
        // Old format: make an events array with previous data + new event
        $decoded = [
            'events' => [$decoded, $newDetails],
        ];
    }

    return json_encode($decoded, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
}

/**
 * Helper: update order payment_status + payment_details safely
 * Only updates payment_status if current status is NULL / '' / 'pending'
 *
 * @param mysqli        $con
 * @param int|string    $orderId
 * @param string        $newStatus   'success' | 'pending' | 'cancelled'
 * @param string|null   $sessionId
 * @param array|null    $details     single event details array to append
 */
function updateOrderPaymentStatus(mysqli $con, $orderId, string $newStatus, ?string $sessionId = null, ?array $details = null): void
{
    $orderId = (int) $orderId;
    if ($orderId <= 0) {
        return;
    }

    // 1) Get current status + existing payment_details
    $stmt = $con->prepare("SELECT payment_status, payment_details FROM orders WHERE order_id = ?");
    if (!$stmt) {
        error_log('Webhook: failed to prepare SELECT: ' . $con->error);
        return;
    }
    $stmt->bind_param('i', $orderId);
    $stmt->execute();
    $res = $stmt->get_result();
    $row = $res->fetch_assoc();
    $stmt->close();

    if (!$row) {
        error_log("Webhook: order_id {$orderId} not found");
        return;
    }

    $currentStatus   = $row['payment_status'];
    $existingDetails = $row['payment_details'] ?? null;

    // Merge old + new details (build array of all events)
    $mergedJsonDetails = mergePaymentDetails($existingDetails, $details);

    // Only update status if not finalized
    $canUpdateStatus = (
        $currentStatus === null ||
        $currentStatus === '' ||
        strtolower($currentStatus) === 'pending'
    );

    // 2) Decide SQL depending on what we can update
    if ($canUpdateStatus) {
        // We can change payment_status + optionally session_id + payment_details
        if ($sessionId !== null && $mergedJsonDetails !== null) {
            $stmt2 = $con->prepare("
                UPDATE orders
                SET payment_status = ?, session_id = ?, payment_details = ?
                WHERE order_id = ?
            ");
            if (!$stmt2) {
                error_log('Webhook: failed UPDATE (status+session+details): ' . $con->error);
                return;
            }
            $stmt2->bind_param('sssi', $newStatus, $sessionId, $mergedJsonDetails, $orderId);

        } elseif ($sessionId !== null && $mergedJsonDetails === null) {
            $stmt2 = $con->prepare("
                UPDATE orders
                SET payment_status = ?, session_id = ?
                WHERE order_id = ?
            ");
            if (!$stmt2) {
                error_log('Webhook: failed UPDATE (status+session): ' . $con->error);
                return;
            }
            $stmt2->bind_param('ssi', $newStatus, $sessionId, $orderId);

        } elseif ($sessionId === null && $mergedJsonDetails !== null) {
            $stmt2 = $con->prepare("
                UPDATE orders
                SET payment_status = ?, payment_details = ?
                WHERE order_id = ?
            ");
            if (!$stmt2) {
                error_log('Webhook: failed UPDATE (status+details): ' . $con->error);
                return;
            }
            $stmt2->bind_param('ssi', $newStatus, $mergedJsonDetails, $orderId);

        } else {
            $stmt2 = $con->prepare("
                UPDATE orders
                SET payment_status = ?
                WHERE order_id = ?
            ");
            if (!$stmt2) {
                error_log('Webhook: failed UPDATE (status only): ' . $con->error);
                return;
            }
            $stmt2->bind_param('si', $newStatus, $orderId);
        }

    } else {
        // Status already final (success/cancelled/etc)
        // → only update payment_details and/or session_id
        if ($sessionId !== null && $mergedJsonDetails !== null) {
            $stmt2 = $con->prepare("
                UPDATE orders
                SET session_id = ?, payment_details = ?
                WHERE order_id = ?
            ");
            if (!$stmt2) {
                error_log('Webhook: failed UPDATE (final, session+details): ' . $con->error);
                return;
            }
            $stmt2->bind_param('ssi', $sessionId, $mergedJsonDetails, $orderId);

        } elseif ($sessionId !== null && $mergedJsonDetails === null) {
            $stmt2 = $con->prepare("
                UPDATE orders
                SET session_id = ?
                WHERE order_id = ?
            ");
            if (!$stmt2) {
                error_log('Webhook: failed UPDATE (final, session only): ' . $con->error);
                return;
            }
            $stmt2->bind_param('si', $sessionId, $orderId);

        } elseif ($sessionId === null && $mergedJsonDetails !== null) {
            $stmt2 = $con->prepare("
                UPDATE orders
                SET payment_details = ?
                WHERE order_id = ?
            ");
            if (!$stmt2) {
                error_log('Webhook: failed UPDATE (final, details only): ' . $con->error);
                return;
            }
            $stmt2->bind_param('si', $mergedJsonDetails, $orderId);
        } else {
            // Nothing to update
            return;
        }
    }

    $stmt2->execute();
    $stmt2->close();

    error_log("Webhook: order_id {$orderId} updated to {$newStatus} (canUpdateStatus=" . ($canUpdateStatus ? '1' : '0') . ")");
}

$type = $event->type;

switch (true) {

    // 🔹 Handle ALL checkout.session.* events
    case (strpos($type, 'checkout.session.') === 0):
        /** @var \Stripe\Checkout\Session $session */
        $session   = $event->data->object;
        $orderId   = $session->metadata->order_id ?? null;
        $sessionId = $session->id;

        // Try to get full PaymentIntent for card details (if exists)
        $pi = null;
        if (!empty($session->payment_intent)) {
            $pi = \Stripe\PaymentIntent::retrieve($session->payment_intent);
        }

        // Build detailed event entry
        $details = buildPaymentDetails($pi, $session, $type);

        // Decide local payment_status based on event type + Stripe status
        $payStatus = $session->payment_status ?? null; // 'paid' | 'unpaid' | 'no_payment_required'
        if ($type === 'checkout.session.completed' || $type === 'checkout.session.async_payment_succeeded') {
            if ($payStatus === 'paid' || $session->status === 'complete') {
                $newStatus = 'success';
            } else {
                $newStatus = 'pending';
            }
        } elseif ($type === 'checkout.session.expired' || $type === 'checkout.session.async_payment_failed') {
            $newStatus = 'cancelled';
        } else {
            // Any other checkout.session.* → keep as pending
            $newStatus = 'pending';
        }

        if ($orderId) {
            updateOrderPaymentStatus($con, $orderId, $newStatus, $sessionId, $details);
        }
        break;

    // 🔹 Handle ALL payment_intent.* events
    case (strpos($type, 'payment_intent.') === 0):
        /** @var \Stripe\PaymentIntent $pi */
        $pi      = $event->data->object;
        $orderId = $pi->metadata->order_id ?? null;

        // Build details for this event
        $details = buildPaymentDetails($pi, null, $type);

        if ($orderId) {
            $piStatus = $pi->status;   // 'succeeded', 'processing', 'requires_payment_method', 'canceled', ...

            if ($piStatus === 'succeeded') {
                $newStatus = 'success';
            } elseif (in_array($piStatus, ['canceled', 'requires_payment_method'], true)) {
                $newStatus = 'cancelled';
            } elseif (in_array($piStatus, [
                'processing',
                'requires_action',
                'requires_capture',
                'requires_confirmation'
            ], true)) {
                $newStatus = 'pending';
            } else {
                $newStatus = 'pending';
            }

            updateOrderPaymentStatus($con, $orderId, $newStatus, null, $details);
        }
        break;

    // 🔹 Any other Stripe event → try to attach details if we can find order_id
    default:
        $object  = $event->data->object;
        $orderId = null;

        // 1) Try metadata->order_id directly on this object
        if (isset($object->metadata) && isset($object->metadata->order_id)) {
            $orderId = $object->metadata->order_id;
        }
        // 2) Or if it references a payment_intent, try to fetch that and get its metadata->order_id
        elseif (isset($object->payment_intent) && $object->payment_intent) {
            try {
                $pi = \Stripe\PaymentIntent::retrieve($object->payment_intent);
                $orderId = $pi->metadata->order_id ?? null;
            } catch (Throwable $e) {
                error_log('Webhook: error retrieving PI in default case: ' . $e->getMessage());
            }
        }

        // Build a minimal details object for this generic event
        $details = [
            'source'      => 'stripe_webhook',
            'event_type'  => $type,
            'created_at'  => time(),
            'object_id'   => $object->id ?? null,
            'raw_summary' => [
                'object_type' => $object->object ?? null,
                'status'      => $object->status ?? null,
            ],
        ];

        if ($orderId) {
            // We pass 'pending' here but:
            //  - updateOrderPaymentStatus will ONLY change payment_status
            //    if it is NULL / '' / 'pending'
            //  - if status is already success/cancelled, it will ONLY append
            //    payment_details without changing the status ✅
            updateOrderPaymentStatus($con, $orderId, 'pending', null, $details);
        }

        error_log('Webhook: generic handled event type ' . $type);
        break;
}

// All good
http_response_code(200);
echo json_encode(['received' => true]);

