<?php
require "../connect.php";
require "../lang.php";

header('Content-Type: application/json; charset=utf-8');
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/error.log');

// Start timing
$start_time = microtime(true);

if (!$con) {
    error_log('Database connection failed: ' . mysqli_connect_error());
    http_response_code(500);
    echo json_encode(['status' => 500, 'message' => 'Database connection failed']);
    exit;
}

// Check if embedding column exists
$check_column = $con->query("SHOW COLUMNS FROM products LIKE 'embedding'");
if ($check_column->num_rows === 0) {
    error_log('Embedding column missing in products table');
    http_response_code(500);
    echo json_encode(['status' => 500, 'message' => 'Embedding column missing']);
    exit;
}

$api_key = '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';
$lang = get_current_language();

// Cosine similarity function
function cosineSimilarity($vecA, $vecB) {
    if (count($vecA) !== count($vecB) || empty($vecA) || empty($vecB)) {
        error_log('Cosine similarity error: Vector lengths - A: ' . count($vecA) . ', B: ' . count($vecB));
        return 0;
    }
    $dotProduct = 0.0;
    $normA = 0.0;
    $normB = 0.0;
    for ($i = 0; $i < count($vecA); $i++) {
        $dotProduct += $vecA[$i] * $vecB[$i];
        $normA += $vecA[$i] * $vecA[$i];
        $normB += $vecB[$i] * $vecB[$i];
    }
    $normA = sqrt($normA);
    $normB = sqrt($normB);
    return ($normA * $normB == 0) ? 0 : $dotProduct / ($normA * $normB);
}

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405);
    echo json_encode(['status' => 405, 'message' => 'Method Not Allowed']);
    exit;
}

// Get parameters
$search = $_GET['search'] ?? '';
$category_id = $_GET['category_id'] ?? '';
$location_id = $_GET['location_id'] ?? '';
$from_price = $_GET['from_price'] ?? '';
$to_price = $_GET['to_price'] ?? '';
$order = (int)($_GET['order'] ?? 1);
$user_id = $_GET['user_id'] ?? '';
$limit = min((int)($_GET['limit'] ?? 5), 50);

if (empty($search)) {
    http_response_code(400);
    echo json_encode(['status' => 400, 'message' => 'Search query is required']);
    exit;
}

// Cache directory
$cache_dir = __DIR__ . '/cache';
if (!is_dir($cache_dir)) {
    mkdir($cache_dir, 0755, true);
}

// Get keywords from ChatGPT
$cache_key = md5($search);
$keyword_cache_file = $cache_dir . '/' . $cache_key . '_keywords.json';
$keywords = [];

if (file_exists($keyword_cache_file) && (time() - filemtime($keyword_cache_file)) < 3600) {
    $keywords = json_decode(file_get_contents($keyword_cache_file), true);
    error_log('Using cached keywords for query: ' . $search);
} else {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, 'https://api.openai.com/v1/chat/completions');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Bearer ' . $api_key,
        'Content-Type: application/json'
    ]);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
        'model' => 'gpt-3.5-turbo',
        'messages' => [
            [
                'role' => 'user',
                'content' => 'Given the search query "' . $search . '", return a JSON array of relevant keywords or a category name that best describes the product type. Focus on product categories or descriptive terms suitable for an e-commerce platform. For example, for "ساعة ضد المي", return ["waterproof watch", "water-resistant watch", "sports watch", "watch"] or "Watches". Respond only with the JSON array or string.'
            ]
        ],
        'max_tokens' => 50
    ]));
    curl_setopt($ch, CURLOPT_TIMEOUT, 6);

    $keyword_response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);

    if ($http_code === 200 && !$curl_error) {
        $keyword_data = json_decode($keyword_response, true);
        if (isset($keyword_data['choices'][0]['message']['content'])) {
            $keywords = json_decode($keyword_data['choices'][0]['message']['content'], true);
            if (!is_array($keywords)) {
                $keywords = [$keywords];
            }
            file_put_contents($keyword_cache_file, json_encode($keywords));
            error_log('Keywords for query "' . $search . '": ' . json_encode($keywords));
        } else {
            $keywords = [$search];
        }
    } else {
        error_log('ChatGPT API error: HTTP ' . $http_code . ', cURL error: ' . $curl_error);
        $keywords = [$search];
    }
}

// Split keywords into individual terms
$split_keywords = [];
foreach ($keywords as $keyword) {
    $terms = preg_split('/\s+/', strtolower(trim($keyword)));
    $split_keywords = array_merge($split_keywords, $terms);
}
$split_keywords = array_unique($split_keywords);

// Cache query embedding
$cache_file = $cache_dir . '/' . $cache_key . '.json';
if (file_exists($cache_file) && (time() - filemtime($cache_file)) < 3600) {
    $query_embedding = json_decode(file_get_contents($cache_file), true);
    error_log('Using cached embedding for query: ' . $search);
} else {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, 'https://api.openai.com/v1/embeddings');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Bearer ' . $api_key,
        'Content-Type: application/json'
    ]);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
        'model' => 'text-embedding-ada-002',
        'input' => $search
    ]));
    curl_setopt($ch, CURLOPT_TIMEOUT, 6);

    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curl_error = curl_error($ch);
    curl_close($ch);

    if ($http_code !== 200 || $curl_error) {
        error_log('OpenAI API error: HTTP ' . $http_code . ', cURL error: ' . $curl_error);
        http_response_code(500);
        echo json_encode(['status' => 500, 'message' => 'OpenAI API error']);
        exit;
    }

    $response_data = json_decode($response, true);
    if (!isset($response_data['data'][0]['embedding'])) {
        error_log('Invalid OpenAI response: ' . json_last_error_msg());
        http_response_code(500);
        echo json_encode(['status' => 500, 'message' => 'Invalid embedding data']);
        exit;
    }

    $query_embedding = $response_data['data'][0]['embedding'];
    file_put_contents($cache_file, json_encode($query_embedding));
}

// Build SQL query
$sql = "SELECT p.*, 
        CASE WHEN f.favorite_id IS NOT NULL THEN 1 ELSE 0 END as is_favorite,
        (
            SELECT JSON_ARRAYAGG(pi.url)
            FROM product_images pi
            WHERE pi.product_id = p.product_id
        ) as images,
        p.embedding
        FROM products p
        LEFT JOIN favorites f ON p.product_id = f.favorite_productId AND f.favorite_userId = ?
        WHERE p.hidden = '0'";
$bind_types = "i";
$bind_values = [$user_id ? (int)$user_id : 0];

// Keyword filter
$keyword_filter = '';
if (!empty($split_keywords)) {
    $keyword_conditions = [];
    foreach ($split_keywords as $keyword) {
        $keyword = $con->real_escape_string($keyword);
        $keyword_conditions[] = "LOWER(p.product_name) LIKE '%$keyword%' OR LOWER(p.product_name_en) LIKE '%$keyword%' OR LOWER(p.product_description) LIKE '%$keyword%' OR LOWER(p.description_en) LIKE '%$keyword%'";
    }
    $keyword_filter = ' AND (' . implode(' OR ', $keyword_conditions) . ')';
}
$sql .= $keyword_filter;

// Additional filters
if (!empty($category_id)) {
    $sql .= " AND p.product_catId = ?";
    $bind_types .= "i";
    $bind_values[] = (int)$category_id;
}
if (!empty($location_id)) {
    $sql .= " AND p.location_id = ?";
    $bind_types .= "i";
    $bind_values[] = (int)$location_id;
}
if ($from_price !== '') {
    $sql .= " AND p.product_price >= ?";
    $bind_types .= "d";
    $bind_values[] = (float)$from_price;
}
if ($to_price !== '') {
    $sql .= " AND p.product_price <= ?";
    $bind_types .= "d";
    $bind_values[] = (float)$to_price;
}

// Ordering
switch ($order) {
    case 2:
        $sql .= " ORDER BY p.product_price ASC";
        break;
    case 3:
        $sql .= " ORDER BY p.product_price DESC";
        break;
    case 1:
    default:
        $sql .= " ORDER BY p.created_at DESC";
        break;
}

// Prepare and execute query
$stmt = $con->prepare($sql);
if (!$stmt) {
    error_log('Prepare failed: ' . $con->error);
    http_response_code(500);
    echo json_encode(['status' => 500, 'message' => 'Query preparation failed']);
    exit;
}

if (!empty($bind_values)) {
    $stmt->bind_param($bind_types, ...$bind_values);
}
$stmt->execute();
$result = $stmt->get_result();

$products = [];
$skipped = 0;
while ($row = $result->fetch_assoc()) {
    if (!empty($row['embedding'])) {
        $embedding = json_decode($row['embedding'], true);
        if (json_last_error() === JSON_ERROR_NONE && is_array($embedding) && count($embedding) === 1536) {
            $similarity = cosineSimilarity($query_embedding, $embedding);
            if ($similarity >= 0.4) {
                $row['similarity'] = $similarity;
                $row['images'] = !empty($row['images']) ? json_decode($row['images'], true) : [];
                $row['product_name'] = $lang === 'ar' ? $row['product_name'] : $row['product_name_en'];
                $row['product_description'] = $lang === 'ar' ? $row['product_description'] : $row['description_en'];
                unset($row['product_name_en'], $row['description_en'], $row['embedding']);
                $products[] = $row;
            } else {
                error_log('Product ID ' . $row['product_id'] . ' skipped due to low similarity: ' . $similarity);
                $skipped++;
            }
        } else {
            error_log('Invalid embedding for product ID ' . $row['product_id'] . ': ' . json_last_error_msg());
            $skipped++;
        }
    } else {
        error_log('Empty embedding for product ID ' . $row['product_id']);
        $skipped++;
    }
}

$stmt->close();
$con->close();

$execution_time = microtime(true) - $start_time;
error_log("Processed query '$search' in $execution_time seconds, skipped $skipped products");

// Fallback if no products found
if (empty($products)) {
    $sql = "SELECT p.*, 
            CASE WHEN f.favorite_id IS NOT NULL THEN 1 ELSE 0 END as is_favorite,
            (
                SELECT JSON_ARRAYAGG(pi.url)
                FROM product_images pi
                WHERE pi.product_id = p.product_id
            ) as images,
            p.embedding
            FROM products p
            LEFT JOIN favorites f ON p.product_id = f.favorite_productId AND f.favorite_userId = ?
            WHERE p.hidden = '0' AND p.embedding IS NOT NULL AND p.embedding != ''";
    $bind_types = "i";
    $bind_values = [$user_id ? (int)$user_id : 0];

    if (!empty($category_id)) {
        $sql .= " AND p.product_catId = ?";
        $bind_types .= "i";
        $bind_values[] = (int)$category_id;
    }
    if (!empty($location_id)) {
        $sql .= " AND p.location_id = ?";
        $bind_types .= "i";
        $bind_values[] = (int)$location_id;
    }
    if ($from_price !== '') {
        $sql .= " AND p.product_price >= ?";
        $bind_types .= "d";
        $bind_values[] = (float)$from_price;
    }
    if ($to_price !== '') {
        $sql .= " AND p.product_price <= ?";
        $bind_types .= "d";
        $bind_values[] = (float)$to_price;
    }

    switch ($order) {
        case 2:
            $sql .= " ORDER BY p.product_price ASC";
            break;
        case 3:
            $sql .= " ORDER BY p.product_price DESC";
            break;
        case 1:
        default:
            $sql .= " ORDER BY p.created_at DESC";
            break;
    }

    $stmt = $con->prepare($sql);
    if ($stmt) {
        if (!empty($bind_values)) {
            $stmt->bind_param($bind_types, ...$bind_values);
        }
        $stmt->execute();
        $result = $stmt->get_result();
        while ($row = $result->fetch_assoc()) {
            $embedding = json_decode($row['embedding'], true);
            if (json_last_error() === JSON_ERROR_NONE && is_array($embedding) && count($embedding) === 1536) {
                $similarity = cosineSimilarity($query_embedding, $embedding);
                if ($similarity >= 0.4) {
                    $row['similarity'] = $similarity;
                    $row['images'] = !empty($row['images']) ? json_decode($row['images'], true) : [];
                    $row['product_name'] = $lang === 'ar' ? $row['product_name'] : $row['product_name_en'];
                    $row['product_description'] = $lang === 'ar' ? $row['product_description'] : $row['description_en'];
                    unset($row['product_name_en'], $row['description_en'], $row['embedding']);
                    $products[] = $row;
                } else {
                    error_log('Product ID ' . $row['product_id'] . ' skipped due to low similarity: ' . $similarity);
                    $skipped++;
                }
            } else {
                error_log('Invalid embedding for product ID ' . $row['product_id'] . ': ' . json_last_error_msg());
                $skipped++;
            }
        }
        $stmt->close();
    }
}

// Return response
if (empty($products)) {
    echo json_encode([
        'status' => 200,
        'message' => 'No products found',
        'query' => $search,
        'keywords' => $keywords,
        'results' => [],
        'skipped' => $skipped,
        'execution_time' => $execution_time
    ]);
} else {
    usort($products, function($a, $b) {
        return $b['similarity'] <=> $a['similarity'];
    });
    $products = array_slice($products, 0, $limit);
    echo json_encode([
        'status' => 200,
        'message' => 'Products retrieved successfully',
        'query' => $search,
        'keywords' => $keywords,
        'results' => $products,
        'execution_time' => $execution_time
    ]);
}
?>