<?php

ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__.'/cart-error.log');
error_reporting(E_ALL);

function jsonResponse(array $data, int $code = 200) {
    http_response_code($code);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode($data, JSON_UNESCAPED_UNICODE);
    exit;
}
function apiError(string $msg, int $code = 400) {
    jsonResponse(['success'=>false, 'message'=>$msg], $code);
}
function apiSuccess($payload = [], string $msg = 'Success') {
    jsonResponse(['success'=>true, 'message'=>$msg, 'data'=>$payload]);
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    apiError('Use POST', 405);
}

$input = file_get_contents('php://input');
if ($input === false) apiError('Cannot read input');
$data = json_decode($input, true);
if (json_last_error() !== JSON_ERROR_NONE) apiError('Invalid JSON: '.json_last_error_msg());

$action  = strtolower(trim($data['action'] ?? ''));
$user_id = trim($data['user_id'] ?? '');
if ($action === '')  apiError('action is required');
if ($user_id === '') apiError('user_id is required');

require __DIR__.'/../connect.php';
if (!$con || $con->connect_error) {
    error_log('DB connect: '.$con->connect_error);
    apiError('Database unavailable', 500);
}


$stmtFind = $con->prepare("SELECT item_id, item_quantity FROM cart WHERE item_userId = ? AND item_productId = ? LIMIT 1");
$stmtIns  = $con->prepare("INSERT INTO cart (item_userId, item_productId, item_quantity, added_at) VALUES (?, ?, ?, NOW())");
$stmtUpd  = $con->prepare("UPDATE cart SET item_quantity = ? WHERE item_id = ?");
$stmtDel  = $con->prepare("DELETE FROM cart WHERE item_userId = ? AND item_productId = ?");

if (!$stmtFind || !$stmtIns || !$stmtUpd || !$stmtDel) {
    $err = $con->error;
    error_log("Prepare failed: $err");
    apiError("Prepare failed: $err", 500);
}


$items = $data['items'] ?? null;
if (!$items && in_array($action, ['add','increase','decrease','set','remove'])) {
    $pid = $data['product_id'] ?? null;
    $qty = (int)($data['quantity'] ?? 1);
    if ($pid === null) apiError("product_id required for '$action'");
    $items = [['product_id'=>(int)$pid, 'quantity'=>$qty]];
}


if (in_array($action, ['add','increase','decrease','set','remove'])) {
    if (!is_array($items) || empty($items)) apiError('items empty');

    $con->autocommit(false);
    try {
        foreach ($items as $it) {
            $pid = (int)($it['product_id'] ?? 0);
            $qty = (int)($it['quantity'] ?? 1);
            if ($pid <= 0) throw new Exception("Invalid product_id: $pid");

            
            $stmtFind->bind_param('si', $user_id, $pid);
            if (!$stmtFind->execute()) {
                throw new Exception("Find execute failed: " . $stmtFind->error);
            }
            $stmtFind->store_result();
            $found = $stmtFind->num_rows > 0;
            $stmtFind->bind_result($item_id, $cur_qty);
            if ($found) $stmtFind->fetch();
            $stmtFind->free_result();

            if ($action === 'remove') {
                $stmtDel->bind_param('si', $user_id, $pid);
                if (!$stmtDel->execute()) {
                    throw new Exception("Remove failed: " . $stmtDel->error);
                }
                continue;
            }

            if (!$found) {
                if ($action === 'decrease') continue;
                $new = ($action === 'set') ? max(0, $qty) : max(1, $qty);
                if ($new <= 0) {
                    $stmtDel->bind_param('si', $user_id, $pid);
                    $stmtDel->execute();
                    continue;
                }
                $stmtIns->bind_param('sii', $user_id, $pid, $new);
                if (!$stmtIns->execute()) {
                    throw new Exception("Insert failed: " . $stmtIns->error);
                }
                continue;
            }

            
            if ($action === 'add' || $action === 'increase') {
                $new = $cur_qty + max(1, $qty);
            } elseif ($action === 'decrease') {
                $new = $cur_qty - max(1, $qty);
            } elseif ($action === 'set') {
                $new = max(0, $qty);
            }

            if ($new <= 0) {
                $stmtDel->bind_param('si', $user_id, $pid);
                if (!$stmtDel->execute()) throw new Exception("Delete failed: " . $stmtDel->error);
            } else {
                $stmtUpd->bind_param('ii', $new, $item_id);
                if (!$stmtUpd->execute()) throw new Exception("Update failed: " . $stmtUpd->error);
            }
        }
        $con->commit();
    } catch (Throwable $e) {
        $con->rollback();
        $errorMsg = $e->getMessage();
        error_log("TRANSACTION ERROR: $errorMsg");
        apiError("Transaction failed: $errorMsg", 500);  
    }
    $con->autocommit(true);
}

elseif ($action === 'clear') {
    $stmt = $con->prepare("DELETE FROM cart WHERE item_userId = ?");
    if (!$stmt) apiError("Clear prepare failed: " . $con->error, 500);
    $stmt->bind_param('s', $user_id);
    $stmt->execute();
    $stmt->close();
}

elseif (!in_array($action, ['list','select'])) {
    apiError("Invalid action: $action");
}


$cart = []; $total_items = 0; $total_qty = 0; $total_cost = 0.0;

$q = "
    SELECT c.item_id, c.item_quantity, c.item_productId, c.added_at,
           p.product_name, p.product_price, p.product_discount, p.product_image
    FROM cart c
    LEFT JOIN products p ON p.product_id = c.item_productId
    WHERE c.item_userId = ?
    ORDER BY c.added_at DESC
";
$stmt = $con->prepare($q);
if (!$stmt) {
    error_log("SELECT prepare failed: " . $con->error);
    apiError("SELECT failed: " . $con->error, 500);
}
$stmt->bind_param('s', $user_id);
$stmt->execute();
$res = $stmt->get_result();

while ($row = $res->fetch_assoc()) {
    $price = (float)($row['product_price'] ?? 0);
    $disc  = (float)($row['product_discount'] ?? 0);
    $qty   = (int)$row['item_quantity'];
    $after = max(0, $price - $disc);
    $line  = $after * $qty;

    $cart[] = [
        'item_id'          => (int)$row['item_id'],
        'product_id'       => (int)$row['item_productId'],
        'quantity'         => $qty,
        'added_at'         => $row['added_at'],
        'product_name'     => $row['product_name'] ?? 'Unknown',
        'product_image'    => $row['product_image'] ?? '',
        'unit_price'       => $price,
        'discount'          => $disc,
        'unit_after_disc'  => $after,
        'line_total'       => $line,
    ];
    $total_items++;
    $total_qty   += $qty;
    $total_cost  += $line;
}
$stmt->close();

apiSuccess([
    'user_id'     => $user_id,
    'action'      => $action,
    'total_items' => $total_items,
    'total_qty'   => $total_qty,
    'total_cost'  => round($total_cost, 2),
    'items'       => $cart
], 'Cart processed');