<?php
require "../../connect.php";
require "../../helpers/jwt.php";

header('Content-Type: application/json; charset=utf-8');

$user = getAuthorizedUser();

if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405);
    exit(json_encode("Method Not Allowed"));
}

$page     = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$per_page = isset($_GET['per_page']) ? min(100, max(1, (int)$_GET['per_page'])) : 10;
$offset   = ($page - 1) * $per_page;

// ----------------------
// COUNT
// ----------------------
$countSql = "SELECT COUNT(*) AS total FROM products";
$countRes = $con->query($countSql);
if (!$countRes) {
    http_response_code(500);
    exit(json_encode([
        'error'   => 'Count query failed',
        'details' => $con->error,
    ], JSON_UNESCAPED_UNICODE));
}

$total       = (int)$countRes->fetch_assoc()['total'];
$total_pages = (int)ceil($total / $per_page);

// ----------------------
// MAIN PRODUCTS QUERY
// ----------------------
$sql = "
  SELECT
    p.*,
    c.cat_name       AS category_name,
    sc.cat_name      AS subcategory_name,
    u.user_name      AS seller_name,
    u.user_image     AS seller_image,
    u.user_email     AS seller_email,
    a.name           AS address_name,
    l.location_name  AS location_name,
    c.cat_id         AS category_id,
    pkg.package_text AS package_name
  FROM products p
  LEFT JOIN categories c  ON p.product_catId    = c.cat_id AND c.parent_id = 0
  LEFT JOIN categories sc ON p.product_subCatId = sc.cat_id AND sc.parent_id != 0
  LEFT JOIN users u       ON p.seller_id        = u.user_id
  LEFT JOIN addresses a   ON p.address_id       = a.address_id
  LEFT JOIN locations l   ON p.location_id      = l.location_id
  LEFT JOIN packages pkg  ON p.package_id       = pkg.package_id
  ORDER BY p.product_id DESC
  LIMIT {$per_page} OFFSET {$offset}
";

$res = $con->query($sql);
if (!$res) {
    http_response_code(500);
    exit(json_encode([
        'error'   => 'Products query failed',
        'details' => $con->error,
    ], JSON_UNESCAPED_UNICODE));
}

// ----------------------
// BUILD BASE RESPONSE
// ----------------------
$products   = [];
$productIds = [];

while ($row = $res->fetch_assoc()) {
    $pid = (int)$row['product_id'];
    $productIds[] = $pid;

    $row['images']  = [];
    $row['reviews'] = [];

    // ✅ Colors normalization (from products.product_colors JSON)
    $colorsRaw = $row['product_colors'] ?? '[]';
    $decodedColors = [];

    if (is_string($colorsRaw)) {
        $tmp = json_decode($colorsRaw, true);
        $decodedColors = is_array($tmp) ? $tmp : [];
    }

    // Keep a consistent key name for API output
    $row['colors'] = $decodedColors;

    // ✅ Sizes/options will come from product_props
    $row['sizes'] = [];

    // Optional: hide raw DB field
    // unset($row['product_colors']);

    $products[$pid] = $row;
}

// ----------------------
// EMPTY
// ----------------------
if (empty($productIds)) {
    http_response_code(200);
    echo json_encode([
        'data' => [],
        'meta' => [
            'total'        => $total,
            'per_page'     => $per_page,
            'current_page' => $page,
            'total_pages'  => $total_pages,
            'next_page'    => $page < $total_pages ? "?page=" . ($page + 1) . "&per_page={$per_page}" : null,
            'prev_page'    => $page > 1 ? "?page=" . ($page - 1) . "&per_page={$per_page}" : null,
        ],
    ], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    exit;
}

$idList = implode(',', array_map('intval', $productIds));

// ----------------------
// IMAGES (batch)
// ----------------------
$sqlImg = "SELECT product_id, url FROM product_images WHERE product_id IN ($idList)";
if ($r2 = $con->query($sqlImg)) {
    while ($img = $r2->fetch_assoc()) {
        $pid = (int)$img['product_id'];
        $url = $img['url'] ?? '';

        if ($url && !str_starts_with($url, 'https')) {
            $url = "https://api.tiffanyflowers.ae/admin/" . ltrim($url, '/');
        }

        $products[$pid]['images'][] = $url;
    }
}

// ----------------------
// SIZES / OPTIONS (batch) - from product_props
// ----------------------
$sqlSizes = "SELECT product_id, title, `value` FROM product_props WHERE product_id IN ($idList)";
if ($r3 = $con->query($sqlSizes)) {
    while ($prop = $r3->fetch_assoc()) {
        $pid = (int)$prop['product_id'];
        $products[$pid]['sizes'][] = [
            'title' => $prop['title'] ?? '',
            'value' => $prop['value'] ?? '',
        ];
    }
}

// ----------------------
// REVIEWS (batch)
// ----------------------
$sqlRev = "
  SELECT
    r.review_id,
    r.review_text,
    r.reviewer_name,
    r.user_id          AS reviewer_id,
    u.user_name        AS reviewer_user_name,
    u.user_image       AS reviewer_user_image,
    r.product_id
  FROM product_reviews r
  LEFT JOIN users u ON r.user_id = u.user_id
  WHERE r.product_id IN ($idList)
";

if ($r4 = $con->query($sqlRev)) {
    while ($rev = $r4->fetch_assoc()) {
        $pid = (int)$rev['product_id'];

        $products[$pid]['reviews'][] = [
            'review_id' => (int)($rev['review_id'] ?? 0),
            'text'      => $rev['review_text'] ?? '',
            'reviewer'  => [
                'id'         => (int)($rev['reviewer_id'] ?? 0),
                'name'       => $rev['reviewer_name'] ?? '',
                'user_name'  => $rev['reviewer_user_name'] ?? '',
                'user_image' => $rev['reviewer_user_image'] ?? '',
            ]
        ];
    }
}

// ----------------------
// RESPONSE
// ----------------------
http_response_code(200);
echo json_encode([
    'data' => array_values($products),
    'meta' => [
        'total'        => $total,
        'per_page'     => $per_page,
        'current_page' => $page,
        'total_pages'  => $total_pages,
        'next_page'    => $page < $total_pages ? "?page=" . ($page + 1) . "&per_page={$per_page}" : null,
        'prev_page'    => $page > 1 ? "?page=" . ($page - 1) . "&per_page={$per_page}" : null,
    ],
], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
